<# This Powershell script starts tasks in parallel and waits until all tasks have been finished.
   The tasks are to be specified in a file. If a file is given as parameter, it will be used. 
   If not, file "ParallelTasks.txt" in the current directory will be used. #>

param([string]$File)

<# Default name of file containg the commands for the tasks #>
$global:defaultFileWithTasks = $PSScriptRoot + '\ParallelTasks.txt'

class Task {
    [string]   $cmd
    [string]   $valid
    [System.Diagnostics.Process] $process
    <# An array with all started Tasks #>
    static $allTasks = @()
    <# Wait time in seconds between checking active tasks #>
    static [System.Int32] $waitSeconds = 5
    <# Suppress information every ... seconds #>
    static $suppressPermanentInformation = $true
    <# constructor for a single Task #> 
    Task($cmd) {
      $this.cmd = $cmd
      $this.valid = Test-Path $cmd
      if ($this.valid -eq $true) {
        $this.process = Start-Process -FilePath $this.cmd -PassThru
        [Task]::stdOut("Started: " + $this.cmd)
        [Task]::allTasks += $this
      }
      else {
        throw ("Command file to be started (" + $this.cmd + ") does not exist")
      }
    }
    static createTasksFromFile($file) {
      <# Create the tasks as given in the file (resp. from default file) and start them #>
      if ($file -eq "") {
        $file = $global:defaultFileWithTasks
      }
      if (-not (Test-Path $file)) {
        throw ("'" + $file + "'" + " does not exist")
      }
      [Task]::stdOut("Read tasks from file '" + $file + "'")
      $linesInFile = Get-Content -Path $file
      ForEach ($line In $linesInFile) {
        $t = [Task]::new($line) 
      }
      [Task]::stdOut("Started tasks: " + [Task]::allTasks.count)
    }
    static [void] setSuppressPermanentInformation($newValue) {
      [Task]::suppressPermanentInformation = $newValue
    }
    static [void] setCheckIntervalSeconds($seconds) {
      [Task]::waitSeconds = $seconds
    }
    static [void] stdOut($line) {
      $outmsg = "[" + (Get-Date).ToString('HH:mm:ss') + "] " + $line
      Write-Host $outmsg
    }
    static [void] Wait4EndOfAllTasks() {
      [Task]::stdOut("Wait until all started tasks have been finished. Check every " + [Task]::waitSeconds + " seconds...")
      $previousActiveTasks = 0
      do {
        $activeTasks = 0    
        foreach ($t in [Task]::allTasks) {
          try {
            Get-Process -Id $t.process.Id -ErrorAction Stop | Out-Null 
            $activeTasks += 1
          }
          catch {}  <# when task does not exist anymore #>
        }

        if ($activeTasks -ne 0) {
          if ([Task]::suppressPermanentInformation -eq $false -or $activeTasks -ne $previousActiveTasks) {
            [Task]::stdOut("Active tasks: " + $activeTasks)
            $previousActiveTasks = $activeTasks
          }
          Start-Sleep -Seconds ([Task]::waitSeconds)
        }
      }
      until($activeTasks -eq 0)
    }
}

function main {
  <# The Main function #>	
  <# Use another value for increase/decrease the frequency of checking whether all tasks have been finished: #>
  [Task]::setCheckIntervalSeconds(5)
  <# Use $false for getting information about running tasks every ... seconds: #>
  [Task]::setSuppressPermanentInformation($true)
  try {
    <# Create the tasks from the given file #>
    [Task]::createTasksFromFile($file)
    if ([Task]::allTasks.count -gt 0) {
      [Task]::Wait4EndOfAllTasks()
      [Task]::stdOut("All tasks finished.")
    }
    else {
      [Task]::stdOut("No tasks were started.")
    }
  }
  catch {
    [Task]::stdOut("Error: " + $_)
    exit
  }
}

main 